Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

$GOLD = [System.Drawing.Color]::FromArgb(255, 215, 0)
$BLACK = [System.Drawing.Color]::FromArgb(26, 26, 26)
$DARK_GRAY = [System.Drawing.Color]::FromArgb(42, 42, 42)
$LIGHT_GOLD = [System.Drawing.Color]::FromArgb(255, 244, 204)

$script:SelectedMT4 = $null
$script:MT4Installations = @()

function Get-BrokerName {
    param([string]$originPath)
    
    if (-not $originPath) { return "Unknown Broker" }
    
    $brokerName = $originPath.Trim()
    
    if ($brokerName -like "*\*") {
        $parts = $brokerName -split "\\"
        foreach ($part in $parts) {
            if ($part -and $part -notlike "*.exe" -and $part -ne "Program Files" -and $part -ne "Program Files (x86)") {
                $brokerName = $part
                break
            }
        }
    }
    
    $brokerName = $brokerName -replace "\\terminal\.exe$", ""
    $brokerName = $brokerName -replace "\\terminal64\.exe$", ""
    $brokerName = $brokerName -replace " Terminal$", ""
    $brokerName = $brokerName -replace " MT4$", ""
    $brokerName = $brokerName.Trim()
    
    if ($brokerName.Length -gt 40) {
        $brokerName = $brokerName.Substring(0, 40) + "..."
    }
    
    return $brokerName
}

function Find-MT4Installations {
    $installations = @()
    $appDataTerminal = "$env:APPDATA\MetaQuotes\Terminal"
    
    if (Test-Path $appDataTerminal) {
        $folders = Get-ChildItem -Path $appDataTerminal -Directory -ErrorAction SilentlyContinue
        
        foreach ($folder in $folders) {
            $originFile = Join-Path $folder.FullName "origin.txt"
            $mql4Folder = Join-Path $folder.FullName "MQL4"
            
            if ((Test-Path $originFile) -or (Test-Path $mql4Folder)) {
                try {
                    $brokerName = "Unknown Broker"
                    $originPath = ""
                    
                    if (Test-Path $originFile) {
                        $originPath = (Get-Content $originFile -Raw -ErrorAction SilentlyContinue).Trim()
                        $brokerName = Get-BrokerName -originPath $originPath
                    }
                    
                    $terminalExe = $null
                    if ($originPath -and (Test-Path $originPath)) {
                        if ($originPath -like "*.exe") {
                            $terminalExe = $originPath
                        } else {
                            $testExe = Join-Path $originPath "terminal.exe"
                            if (Test-Path $testExe) { $terminalExe = $testExe }
                            else {
                                $testExe = Join-Path $originPath "terminal64.exe"
                                if (Test-Path $testExe) { $terminalExe = $testExe }
                            }
                        }
                    }
                    
                    $installations += @{
                        Name = $brokerName
                        ExePath = $terminalExe
                        DataFolder = $folder.FullName
                        OriginPath = $originPath
                    }
                } catch {}
            }
        }
    }
    
    return $installations
}

function Install-AxumBot {
    param($selectedInstall)
    
    $installerDir = Split-Path -Parent $PSCommandPath
    
    try {
        $expertsFolder = Join-Path $selectedInstall.DataFolder "MQL4\Experts"
        $profilesFolder = Join-Path $selectedInstall.DataFolder "profiles"
        
        New-Item -ItemType Directory -Force -Path $expertsFolder | Out-Null
        New-Item -ItemType Directory -Force -Path $profilesFolder | Out-Null
        
        $eaSource = Join-Path $installerDir "Axum_Ultimate_Bot_v5_4_LICENSED.ex4"
        
        if (Test-Path $eaSource) {
            $eaDest = Join-Path $expertsFolder "Axum_Ultimate_Bot_v5_4_LICENSED.ex4"
            Copy-Item -Path $eaSource -Destination $eaDest -Force
            $statusEA = "Expert Advisor installed"
        } else {
            return @{ Success = $false; Message = "EA file not found in installer folder" }
        }
        
        $profileSource = Join-Path $installerDir "profiles\AXUM"
        if (Test-Path $profileSource) {
            $profileDest = Join-Path $profilesFolder "AXUM"
            if (Test-Path $profileDest) {
                Remove-Item -Path $profileDest -Recurse -Force
            }
            Copy-Item -Path $profileSource -Destination $profileDest -Recurse -Force
            $statusProfile = "Trading profile installed (6 charts)"
        } else {
            $statusProfile = ""
        }
        
        return @{
            Success = $true
            Message = "Installation Complete!`n`n$statusEA`n$statusProfile"
        }
    } catch {
        return @{
            Success = $false
            Message = "Error: $($_.Exception.Message)"
        }
    }
}

$form = New-Object System.Windows.Forms.Form
$form.Text = "Axum Trading Bot Installer"
$form.Size = New-Object System.Drawing.Size(500, 400)
$form.StartPosition = "CenterScreen"
$form.FormBorderStyle = "FixedSingle"
$form.MaximizeBox = $false
$form.BackColor = $BLACK

$headerPanel = New-Object System.Windows.Forms.Panel
$headerPanel.Size = New-Object System.Drawing.Size(500, 70)
$headerPanel.Location = New-Object System.Drawing.Point(0, 0)
$headerPanel.BackColor = $GOLD

$titleLabel = New-Object System.Windows.Forms.Label
$titleLabel.Text = "AXUM TRADING BOT"
$titleLabel.Font = New-Object System.Drawing.Font("Arial", 20, [System.Drawing.FontStyle]::Bold)
$titleLabel.ForeColor = $BLACK
$titleLabel.BackColor = $GOLD
$titleLabel.AutoSize = $false
$titleLabel.Size = New-Object System.Drawing.Size(500, 35)
$titleLabel.Location = New-Object System.Drawing.Point(0, 10)
$titleLabel.TextAlign = "MiddleCenter"

$subtitleLabel = New-Object System.Windows.Forms.Label
$subtitleLabel.Text = "Installer"
$subtitleLabel.Font = New-Object System.Drawing.Font("Arial", 10)
$subtitleLabel.ForeColor = $BLACK
$subtitleLabel.BackColor = $GOLD
$subtitleLabel.AutoSize = $false
$subtitleLabel.Size = New-Object System.Drawing.Size(500, 20)
$subtitleLabel.Location = New-Object System.Drawing.Point(0, 45)
$subtitleLabel.TextAlign = "MiddleCenter"

$headerPanel.Controls.Add($titleLabel)
$headerPanel.Controls.Add($subtitleLabel)

$infoLabel = New-Object System.Windows.Forms.Label
$infoLabel.Text = "Select your broker:"
$infoLabel.Font = New-Object System.Drawing.Font("Arial", 14, [System.Drawing.FontStyle]::Bold)
$infoLabel.ForeColor = $GOLD
$infoLabel.BackColor = $BLACK
$infoLabel.AutoSize = $false
$infoLabel.Size = New-Object System.Drawing.Size(460, 35)
$infoLabel.Location = New-Object System.Drawing.Point(20, 90)

$mt4Dropdown = New-Object System.Windows.Forms.ComboBox
$mt4Dropdown.Size = New-Object System.Drawing.Size(460, 35)
$mt4Dropdown.Location = New-Object System.Drawing.Point(20, 130)
$mt4Dropdown.DropDownStyle = "DropDownList"
$mt4Dropdown.Font = New-Object System.Drawing.Font("Arial", 12)
$mt4Dropdown.BackColor = [System.Drawing.Color]::White

$statusLabel = New-Object System.Windows.Forms.Label
$statusLabel.Text = ""
$statusLabel.Font = New-Object System.Drawing.Font("Arial", 10)
$statusLabel.ForeColor = $LIGHT_GOLD
$statusLabel.BackColor = $BLACK
$statusLabel.AutoSize = $false
$statusLabel.Size = New-Object System.Drawing.Size(460, 100)
$statusLabel.Location = New-Object System.Drawing.Point(20, 180)

$installButton = New-Object System.Windows.Forms.Button
$installButton.Text = "INSTALL"
$installButton.Size = New-Object System.Drawing.Size(460, 50)
$installButton.Location = New-Object System.Drawing.Point(20, 290)
$installButton.Font = New-Object System.Drawing.Font("Arial", 14, [System.Drawing.FontStyle]::Bold)
$installButton.BackColor = $GOLD
$installButton.ForeColor = $BLACK
$installButton.FlatStyle = "Flat"

$form.Controls.Add($headerPanel)
$form.Controls.Add($infoLabel)
$form.Controls.Add($mt4Dropdown)
$form.Controls.Add($statusLabel)
$form.Controls.Add($installButton)

$statusLabel.Text = "Scanning for MetaTrader 4..."
$form.Refresh()

$script:MT4Installations = Find-MT4Installations

if ($script:MT4Installations.Count -gt 0) {
    foreach ($install in $script:MT4Installations) {
        $mt4Dropdown.Items.Add($install.Name) | Out-Null
    }
    $mt4Dropdown.SelectedIndex = 0
    $statusLabel.Text = "Found $($script:MT4Installations.Count) broker(s).`n`nSelect your broker and click INSTALL."
} else {
    $mt4Dropdown.Items.Add("No MT4 found") | Out-Null
    $mt4Dropdown.SelectedIndex = 0
    $installButton.Enabled = $false
    $statusLabel.Text = "No MetaTrader 4 installations found.`n`nPlease install MT4 from your broker first."
}

$script:Installed = $false

$installButton.Add_Click({
    if ($script:Installed) {
        if ($script:SelectedMT4.ExePath -and (Test-Path $script:SelectedMT4.ExePath)) {
            Start-Process -FilePath $script:SelectedMT4.ExePath -ArgumentList "/profile:AXUM"
            $form.Close()
        } else {
            [System.Windows.Forms.MessageBox]::Show(
                "Please launch MetaTrader 4 manually.`n`nThen go to: File -> Profiles -> AXUM",
                "Launch MT4",
                "OK",
                "Information"
            )
            $form.Close()
        }
        return
    }
    
    if ($script:MT4Installations.Count -eq 0) { return }
    
    $selectedIndex = $mt4Dropdown.SelectedIndex
    if ($selectedIndex -lt 0 -or $selectedIndex -ge $script:MT4Installations.Count) {
        [System.Windows.Forms.MessageBox]::Show("Please select a broker!", "Warning", "OK", "Warning")
        return
    }
    
    $script:SelectedMT4 = $script:MT4Installations[$selectedIndex]
    
    $statusLabel.Text = "Installing..."
    $installButton.Enabled = $false
    $installButton.Text = "INSTALLING..."
    $form.Refresh()
    
    $result = Install-AxumBot -selectedInstall $script:SelectedMT4
    
    if ($result.Success) {
        $script:Installed = $true
        $statusLabel.Text = $result.Message
        $installButton.Text = "LAUNCH MT4"
        $installButton.BackColor = [System.Drawing.Color]::FromArgb(0, 200, 0)
        $installButton.Enabled = $true
        
        [System.Windows.Forms.MessageBox]::Show(
            "Installation successful!`n`nClick 'LAUNCH MT4' to start trading.",
            "Success",
            "OK",
            "Information"
        )
    } else {
        $statusLabel.Text = $result.Message
        $installButton.Text = "INSTALL"
        $installButton.Enabled = $true
    }
})

[void]$form.ShowDialog()
